<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Lights Model
 *
 * @property \App\Model\Table\TitlesTable&\Cake\ORM\Association\BelongsTo $Titles
 * @property \App\Model\Table\CountriesTable&\Cake\ORM\Association\BelongsTo $Countries
 *
 * @method \App\Model\Entity\Light newEmptyEntity()
 * @method \App\Model\Entity\Light newEntity(array $data, array $options = [])
 * @method \App\Model\Entity\Light[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Light get($primaryKey, $options = [])
 * @method \App\Model\Entity\Light findOrCreate($search, ?callable $callback = null, $options = [])
 * @method \App\Model\Entity\Light patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\Light[] patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Light|false save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Light saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Light[]|\Cake\Datasource\ResultSetInterface|false saveMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Light[]|\Cake\Datasource\ResultSetInterface saveManyOrFail(iterable $entities, $options = [])
 * @method \App\Model\Entity\Light[]|\Cake\Datasource\ResultSetInterface|false deleteMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Light[]|\Cake\Datasource\ResultSetInterface deleteManyOrFail(iterable $entities, $options = [])
 *
 * @mixin \Cake\ORM\Behavior\TimestampBehavior
 */
class LightsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('lights');
        $this->setDisplayField('homes');
        $this->setPrimaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Titles', [
            'foreignKey' => 'title_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Countries', [
            'foreignKey' => 'country_id',
            'joinType' => 'INNER',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->decimal('donation_amount')
            ->requirePresence('donation_amount', 'create')
            ->notEmptyString('donation_amount');

        $validator
            ->scalar('message')
            ->requirePresence('message', 'create')
            ->notEmptyString('message');

        $validator
            ->scalar('gift_aid_for')
            ->maxLength('gift_aid_for', 255)
            ->allowEmptyString('gift_aid_for');

        $validator
            ->boolean('gift_aid_understood')
            ->requirePresence('gift_aid_understood', 'create')
            ->notEmptyString('gift_aid_understood');

        $validator->integer('title_id')->notEmptyString('title_id');

        $validator
            ->scalar('first_name')
            ->maxLength('first_name', 255)
            ->requirePresence('first_name', 'create')
            ->notEmptyString('first_name');

        $validator
            ->scalar('surname')
            ->maxLength('surname', 255)
            ->requirePresence('surname', 'create')
            ->notEmptyString('surname');

        $validator
            ->scalar('address_1')
            ->maxLength('address_1', 255)
            ->requirePresence('address_1', 'create')
            ->notEmptyString('address_1');

        $validator
            ->scalar('city')
            ->maxLength('city', 255)
            ->requirePresence('city', 'create')
            ->notEmptyString('city');

        $validator
            ->scalar('county')
            ->maxLength('county', 255)
            ->requirePresence('county', 'create')
            ->notEmptyString('county');

        $validator
            ->scalar('postal_code')
            ->maxLength('postal_code', 255)
            ->requirePresence('postal_code', 'create')
            ->notEmptyString('postal_code');

        $validator->integer('country_id')->notEmptyString('country_id');

        $validator
            ->email('email')
            ->requirePresence('email', 'create')
            ->notEmptyString('email');

        $validator
            ->scalar('email_confirm')
            ->maxLength('email_confirm', 255)
            ->requirePresence('email_confirm', 'create')
            ->notEmptyString('email_confirm');

        $validator->add('email_confirm', 'compareWith', [
            'rule' => ['compareWith', 'email'],
            'message' => 'Email addresses do not match',
        ]);

        $validator
            ->scalar('telephone')
            ->maxLength('telephone', 255)
            ->requirePresence('telephone', 'create')
            ->notEmptyString('telephone');

        $validator->boolean('no_email')->notEmptyString('no_email');

        $validator->boolean('no_telephone')->notEmptyString('no_telephone');

        $validator->boolean('no_letter')->notEmptyString('no_letter');

        $validator->boolean('optin_email')->notEmptyString('optin_email');

        $validator->boolean('optin_telephone')->notEmptyString('optin_telephone');

        $validator->boolean('optin_letter')->notEmptyString('optin_letter');

        $validator->boolean('is_mail_sent')->notEmptyString('is_mail_sent');

        $validator
            ->scalar('decline_code')
            ->maxLength('decline_code', 255)
            ->allowEmptyString('decline_code');

        $validator
            ->scalar('expiry_date')
            ->maxLength('expiry_date', 255)
            ->allowEmptyString('expiry_date');

        $validator
            ->scalar('bank_auth_code')
            ->maxLength('bank_auth_code', 255)
            ->allowEmptyString('bank_auth_code');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn('title_id', 'Titles'), ['errorField' => 'title_id']);
        $rules->add($rules->existsIn('country_id', 'Countries'), ['errorField' => 'country_id']);

        return $rules;
    }

    // beforeSave function
    public function beforeMarshal(\Cake\Event\EventInterface $event, \ArrayObject $data, \ArrayObject $options)
    {
        if (!empty($data['homes'])) {
            $data['homes'] = implode(', ', $data['homes']);
        }
    }
}
