<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Golfs Model
 *
 * @property \App\Model\Table\GolfTypesTable&\Cake\ORM\Association\BelongsTo $GolfTypes
 * @property \App\Model\Table\GolfPlayersTable&\Cake\ORM\Association\HasMany $GolfPlayers
 *
 * @method \App\Model\Entity\Golf newEmptyEntity()
 * @method \App\Model\Entity\Golf newEntity(array $data, array $options = [])
 * @method \App\Model\Entity\Golf[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Golf get($primaryKey, $options = [])
 * @method \App\Model\Entity\Golf findOrCreate($search, ?callable $callback = null, $options = [])
 * @method \App\Model\Entity\Golf patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\Golf[] patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Golf|false save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Golf saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Golf[]|\Cake\Datasource\ResultSetInterface|false saveMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Golf[]|\Cake\Datasource\ResultSetInterface saveManyOrFail(iterable $entities, $options = [])
 * @method \App\Model\Entity\Golf[]|\Cake\Datasource\ResultSetInterface|false deleteMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Golf[]|\Cake\Datasource\ResultSetInterface deleteManyOrFail(iterable $entities, $options = [])
 *
 * @mixin \Cake\ORM\Behavior\TimestampBehavior
 */
class GolfsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('golfs');
        $this->setDisplayField('p1_title_other');
        $this->setPrimaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('GolfTypes', [
            'foreignKey' => 'golf_type_id',
            'joinType' => 'INNER',
        ]);
        $this->hasMany('GolfPlayers', [
            'foreignKey' => 'golf_id',
            'saveStrategy' => 'replace',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator->integer('golf_type_id')->notEmptyString('golf_type_id');

        $validator->boolean('paid')->notEmptyString('paid');

        $validator
            ->scalar('heard_about')
            ->maxLength('heard_about', 255)
            ->requirePresence('heard_about', 'create')
            ->notEmptyString('heard_about');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn('golf_type_id', 'GolfTypes'), ['errorField' => 'golf_type_id']);

        return $rules;
    }

    public function beforeSave($event, $entity, $options)
    {
        $golf_type = \Cake\ORM\TableRegistry::getTableLocator()
            ->get('GolfTypes')
            ->get($entity['golf_type_id']);

        if (count($entity['golf_players']) > $golf_type->players) {
            $entity['golf_players'] = array_slice($entity['golf_players'], 0, $golf_type->players);
        }
    }
}
