<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Registrations Controller
 *
 * @property \App\Model\Table\RegistrationsTable $Registrations
 */
class RegistrationsController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    public function getEventCosts($id)
    {
        $event = $this->Registrations->Events->get($id, contain: ['EventCosts']);

        $costs = $this->Registrations->Events->EventCosts
            ->find()
            ->select(['id', 'name', 'price', 'tickets_available'])
            ->where(['event_id' => (int) $id])
            ->all();

        $cost_hash = [];

        foreach ($costs as $c) {
            $cost_hash[] = [
                'id' => $c->id,
                'name' => $c->name,
                'price' => $c->price,
                'tickets_available' => range(
                    0,
                    min(
                        10,
                        $c->tickets_available !== '' && $c->tickets_available !== null ? $c->tickets_available : 10,
                    ),
                ),
                'sold_out' => $c->tickets_available === 0,
            ];
        }

        $json = json_encode([
            'costs' => $cost_hash,
            'part_reasons' => $event->participation_reasons_on ? true : false,
            'emergency_contact' => $event->emergency_contact_on ? true : false,
            'custom_field_1' => $event->custom_field_1,
            'custom_field_2' => $event->custom_field_2,
            'multiple_tickets' => $event->multiple_tickets ? true : false,
            'dob' => $event->dob_on ? true : false,
            'dob_text' => $event->dob_text,
            'terms' => $event->terms_on ? true : false,
            'terms_text' => $event->terms_text,
        ]);

        $this->response = $this->response->withType('application/json')->withStringBody($json);
        return $this->response;
    }

    public function confirm()
    {
        // $this->loadComponent('SagePay');

        if (!$this->request->getSession()->check('registration_id')) {
            return $this->redirect(['action' => 'register']);
        }

        $registration = $this->Registrations->get(
            $this->request->getSession()->read('registration_id'),
            contain: ['Titles', 'Countries', 'Events' => ['EventCosts'], 'Tickets' => ['EventCosts']],
        );
        $sagepay_url = Configure::read('Forms.SagePay.url');

        // $crypted = $this->SagePay->buildCrypt($registration);

        $crypted = $this->buildCrypt($registration);
        $this->set(compact('registration', 'crypted', 'sagepay_url'));
    }

    public function register()
    {
        $this->loadComponent('Recaptcha.Recaptcha');
        $validation_error = false;

        $registration = $this->Registrations->newEmptyEntity();
        $registration->registration_amount = $this->request->getQuery('amount');
        $registration->gift_aid_understood = $this->request->getQuery('gift_aid_understood');

        if ($this->request->getSession()->check('registration_id') && !$validation_error) {
            $registration = $this->Registrations->get(
                $this->request->getSession()->read('registration_id'),
                contain: ['Titles', 'Countries', 'Events' => ['EventCosts'], 'Tickets' => ['EventCosts']],
            );
        }

        if ($this->request->is(['post', 'put'])) {
            $registration = $this->Registrations->patchEntity($registration, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                $registration = $this->Registrations->patchEntity($registration, $this->request->getData());
                if ($this->Registrations->save($registration)) {
                    // $this->Flash->success(__('The registration has been saved.'));
                    $this->request->getSession()->write('registration_id', $registration->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(__('The registration could not be saved. Please, try again.'));
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        $titles = $this->Registrations->Titles->find('list')->all();
        $countries = $this->Registrations->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $gift_aid_options = Configure::read('Forms.gift_aid_options');

        $google_captcha_key = Configure::read('GoogleCaptcha.key');

        $events = $this->Registrations->Events
            ->find('list')
            ->where(['active' => true])
            ->orderByAsc('name');

        $this->set(compact('registration', 'titles', 'countries', 'gift_aid_options', 'google_captcha_key', 'events'));
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('registration_id')) {
            $registration = $this->Registrations->get(
                $this->request->getSession()->read('registration_id'),
                contain: ['Titles', 'Countries', 'Events' => ['EventCosts'], 'Tickets' => ['EventCosts']],
            );
        } else {
            return $this->redirect(['action' => 'register']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $registration->tx_status = $decoded['Status'];
        $registration->tx_status_detail = $decoded['StatusDetail'];
        $registration->tx_code = $decoded['VendorTxCode'];
        $registration->tx_sagepay_id = $decoded['VPSTxId'];
        $registration->tx_cardtype = $decoded['CardType'];
        $registration->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $registration->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $registration->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $registration->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Registrations->save($registration);
        $this->set(compact('registration', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($registration->is_mail_sent)) {
            $this->_emailConfirmation($registration);
            $this->_emailConfirmation($registration, true);

            $registration->is_mail_sent = true;
            $registration->amount = $decoded['Amount'];
            $this->Registrations->save($registration);
            $this->_subtractTickets($registration);
            $this->request->getSession()->consume('registration_id');
        }
    }

    private function _subtractTickets($registration)
    {
        foreach ($registration->tickets as $ticket) {
            $ticket->event_cost->tickets_available -= $ticket->amount;
            $this->getTableLocator()
                ->get('EventCosts')
                ->save($ticket->event_cost);
        }
    }

    private function _emailConfirmation($registration, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo($admin ? Configure::read('Forms.confirmation_email_recipients.events') : $registration->email)
            ->setSubject("Thank you for registrating for the {$registration->event->name}" . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'registration' => $registration,
            ])
            ->viewBuilder()
            ->setTemplate('event_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($d)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'registrations', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount($d->grandTotal());
        $sagePay->setDescription("{$d->event->name} registration");
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.events'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
