<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Lights Controller
 *
 * @property \App\Model\Table\LightsTable $Lights
 */
class LightsController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    /**
     * View method
     *
     * @param string|null $id Light id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function confirm()
    {
        if (!$this->request->getSession()->check('light_id')) {
            return $this->redirect(['action' => 'donate']);
        }

        $light = $this->Lights->get($this->request->getSession()->read('light_id'), contain: ['Titles', 'Countries']);
        $sagepay_url = Configure::read('Forms.SagePay.url');

        // $crypted = $this->SagePay->buildCrypt($light);

        $crypted = $this->buildCrypt($light);
        $this->set(compact('light', 'crypted', 'sagepay_url'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function donate()
    {
        $this->loadComponent('Recaptcha.Recaptcha');
        $validation_error = false;

        $light = $this->Lights->newEmptyEntity();
        $light->light_amount = $this->request->getQuery('amount');
        $light->gift_aid_understood = $this->request->getQuery('gift_aid_understood');

        if ($this->request->getSession()->check('light_id') && !$validation_error) {
            $light = $this->Lights->get(
                $this->request->getSession()->read('light_id'),
                contain: ['Titles', 'Countries'],
            );
            $light->homes = !empty($light->homes) ? explode(', ', $light->homes) : [];
        }

        if ($this->request->is(['post', 'put'])) {
            $light = $this->Lights->patchEntity($light, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                if ($this->Lights->save($light)) {
                    // $this->Flash->success(__('The light has been saved.'));
                    $this->request->getSession()->write('light_id', $light->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(__('Your registration could not be saved. Please, try again.'));
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        $titles = $this->Lights->Titles->find('list')->all();
        $countries = $this->Lights->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $gift_aid_options = Configure::read('Forms.gift_aid_options');

        $homes = [
            'Mary House' => 'Mary House, Hastings',
            'Martha House' => 'Martha House, Deal',
            'Frances House' => 'Frances House, Deal',
            'No preference' => 'No preference',
        ];

        $google_captcha_key = Configure::read('GoogleCaptcha.key');

        $amounts = $this->fetchTable('Amounts')
            ->find()
            ->all();

        $this->set(
            compact('light', 'homes', 'titles', 'countries', 'gift_aid_options', 'google_captcha_key', 'amounts'),
        );
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('light_id')) {
            $light = $this->Lights->get(
                $this->request->getSession()->read('light_id'),
                contain: ['Titles', 'Countries'],
            );
        } else {
            return $this->redirect(['action' => 'donate']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $light->tx_status = $decoded['Status'];
        $light->tx_status_detail = $decoded['StatusDetail'];
        $light->tx_code = $decoded['VendorTxCode'];
        $light->tx_sagepay_id = $decoded['VPSTxId'];
        $light->tx_cardtype = $decoded['CardType'];
        $light->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $light->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $light->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $light->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Lights->save($light);
        $this->set(compact('light', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($light->is_mail_sent)) {
            $this->_emailConfirmation($light);
            $this->_emailConfirmation($light, true);

            $light->is_mail_sent = true;
            $this->Lights->save($light);
            $this->request->getSession()->consume('light_id');
        }
    }

    private function _emailConfirmation($light, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo($admin ? Configure::read('Forms.confirmation_email_recipients.lights') : $light->email)
            ->setSubject(Configure::read('Forms.Email.subject.lights') . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'light' => $light,
            ])
            ->viewBuilder()
            ->setTemplate('light_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($d)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'lights', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount($d->donation_amount);
        $sagePay->setDescription(Configure::read('Forms.SagePay.item_description.lights'));
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.lights'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
