<?php
declare(strict_types=1);

namespace App\Controller\Admin;

use App\Controller\AppController;

/**
 * Orders Controller
 *
 * @property \App\Model\Table\OrdersTable $Orders
 */
class OrdersController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Orders
            ->find()
            ->contain(['Titles', 'Countries'])
            ->orderByDesc('Orders.created');
        $orders = $this->paginate($query);

        $this->set(compact('orders'));
    }

    /**
     * View method
     *
     * @param string|null $id Order id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $order = $this->Orders->get(
            $id,
            contain: ['Titles', 'Countries', 'DeliveryTitles', 'DeliveryCountries', 'OrderItems' => ['Products']],
        );
        $this->set(compact('order'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $order = $this->Orders->newEmptyEntity();
        if ($this->request->is('post')) {
            $order = $this->Orders->patchEntity($order, $this->request->getData());
            if ($this->Orders->save($order)) {
                $this->Flash->success(__('The order has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The order could not be saved. Please, try again.'));
        }
        $titles = $this->Orders->Titles->find('list', limit: 200)->all();
        $countries = $this->Orders->Countries->find('list', limit: 200)->all();
        $this->set(compact('order', 'titles', 'countries'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Order id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $order = $this->Orders->get(
            $id,
            contain: ['Titles', 'Countries', 'OrderItems', 'DeliveryTitles', 'DeliveryCountries'],
        );
        if ($this->request->is(['patch', 'post', 'put'])) {
            $order = $this->Orders->patchEntity($order, $this->request->getData());
            if ($this->Orders->save($order)) {
                $this->Flash->success(__('The order has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The order could not be saved. Please, try again.'));
        }
        $titles = $this->Orders->Titles->find('list', limit: 200)->all();
        $countries = $this->Orders->Countries->find('list', limit: 200)->all();
        $this->set(compact('order', 'titles', 'countries'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Order id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $order = $this->Orders->get($id);
        if ($this->Orders->delete($order)) {
            $this->Flash->success(__('The order has been deleted.'));
        } else {
            $this->Flash->error(__('The order could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function csv()
    {
        $orders = $this->Orders
            ->find()
            ->contain(['Titles', 'Countries', 'DeliveryTitles', 'DeliveryCountries', 'OrderItems' => ['Products']])
            ->all();

        $header = [
            'Date',
            'Amount',
            'Paid',
            'Tx. Status',
            'Tx. Detail',
            'Tx. code',
            'SagePay ID',
            'Orderd type',
            'Orderd Last digits',
            'Title',
            'Title Other',
            'Name',
            'Surname',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Country',
            'Telephone',
            'Mobile phone',
            'Email',
            'Contact by email',
            'Contact by phone',
            'Contact by letter',
            'Delivery Title',
            'Delivery Title Other',
            'Delivery Name',
            'Delivery Surname',
            'Delivery Address 1',
            'Delivery Address2',
            'Delivery City',
            'Delivery County',
            'Delivery Postal Code',
            'Item',
            'Price',
            'Quantity',
            'Sub-total',
        ];

        // header('Content-Type: text/csv; charset=utf-8');
        // header('Content-Disposition: attachment; filename=orders-' . date('Y-m-d-h-i-s') . '.csv');
        $csv = fopen('php://temp/maxmemory:' . 5 * 1024 * 1024, 'r+');

        fputcsv($csv, $header);

        foreach ($orders as $reg) {
            $row = [
                $reg->created->format('Y-m-d H:i:s'),
                $reg->amount,
                $reg->paid ? 'Yes' : 'No',
                $reg->tx_status,
                $reg->tx_status_detail,
                $reg->tx_code,
                $reg->tx_sagepay_id,
                $reg->tx_cardtype,
                $reg->tx_card_lastdigits,
                $reg->title->name,
                $reg->title_other,
                $reg->first_name,
                $reg->surname,
                $reg->address_1,
                $reg->address_2,
                $reg->city,
                $reg->county,
                $reg->postal_code,
                $reg->country->name,
                $reg->telephone,
                $reg->mobile_phone,
                $reg->email,
                $reg->optin_email ? 'Yes' : 'No',
                $reg->optin_telephone ? 'Yes' : 'No',
                $reg->optin_letter ? 'Yes' : 'No',
                $reg->delivery_title->name,
                $reg->d_title_other,
                $reg->d_first_name,
                $reg->d_surname,
                $reg->d_address_1,
                $reg->d_address_2,
                $reg->d_city,
                $reg->d_county,
                $reg->d_postal_code,
            ];
            foreach ($reg->order_items as $p) {
                if ($p['quantity'] > 0) {
                    $row = array_merge($row, [
                        $p->product->name,
                        number_format((float) $p->price, 2),
                        $p->quantity,
                        $p->subTotal(),
                    ]);
                }
            }

            fputcsv($csv, $row);
        }

        rewind($csv);
        $output = stream_get_contents($csv);

        $response = $this->response;
        $response = $response->withStringBody($output);
        $response = $response->withType('csv');
        $response = $response->withDownload('orders-' . date('Y-m-d-h-i-s') . '.csv');
        return $response;
    }
}
