<?php
declare(strict_types=1);

namespace App\Controller\Admin;

use App\Controller\AppController;

/**
 * Lights Controller
 *
 * @property \App\Model\Table\LightsTable $Lights
 */
class LightsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Lights
            ->find()
            ->contain(['Titles', 'Countries'])
            ->orderByDesc('Lights.created');
        $lights = $this->paginate($query);

        $this->set(compact('lights'));
    }

    /**
     * View method
     *
     * @param string|null $id Light id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $light = $this->Lights->get($id, contain: ['Titles', 'Countries']);
        $this->set(compact('light'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $light = $this->Lights->newEmptyEntity();
        if ($this->request->is('post')) {
            $light = $this->Lights->patchEntity($light, $this->request->getData());
            if ($this->Lights->save($light)) {
                $this->Flash->success(__('The light has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The light could not be saved. Please, try again.'));
        }
        $titles = $this->Lights->Titles->find('list', limit: 200)->all();
        $countries = $this->Lights->Countries->find('list', limit: 200)->all();
        $this->set(compact('light', 'titles', 'countries'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Light id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $light = $this->Lights->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $light = $this->Lights->patchEntity($light, $this->request->getData());
            if ($this->Lights->save($light)) {
                $this->Flash->success(__('The light has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The light could not be saved. Please, try again.'));
        }
        $titles = $this->Lights->Titles->find('list', limit: 200)->all();
        $countries = $this->Lights->Countries->find('list', limit: 200)->all();
        $this->set(compact('light', 'titles', 'countries'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Light id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $light = $this->Lights->get($id);
        if ($this->Lights->delete($light)) {
            $this->Flash->success(__('The light has been deleted.'));
        } else {
            $this->Flash->error(__('The light could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function csv()
    {
        $lights = $this->Lights->find()->contain(['Titles', 'Countries'])->all();

        $header = [
            'Date',
            'Amount',
            'Message',
            'Home preference',
            'Gift Aid Understood',
            'Gift Aid',
            'Title',
            'Title Other',
            'Name',
            'Surname',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Country',
            'Telephone',
            'Mobile phone',
            'Email',
            'Can contact by email',
            'Can contact by phone',
            'Can contact by letter',
            'Tx. Status',
            'Tx. Detail',
            'Tx. code',
            'SagePay ID',
            'Card type',
            'Card Last digits',
        ];

        // header('Content-Type: text/csv; charset=utf-8');
        // header('Content-Disposition: attachment; filename=lights-' . date('Y-m-d-h-i-s') . '.csv');
        $csv = fopen('php://temp/maxmemory:' . 5 * 1024 * 1024, 'r+');

        fputcsv($csv, $header);

        foreach ($lights as $reg) {
            $row = [
                $reg->created->format('Y-m-d H:i:s'),
                $reg->donation_amount,
                $reg->message,
                $reg->homes,
                $reg->gift_aid_understood ? 'Yes' : 'No',
                $reg->gift_aid_for,
                $reg->title->name,
                $reg->title_other,
                $reg->first_name,
                $reg->surname,
                $reg->address_1,
                $reg->address_2,
                $reg->city,
                $reg->county,
                $reg->postal_code,
                $reg->country->name,
                $reg->telephone,
                $reg->mobile_phone,
                $reg->email,
                $reg->optin_email ? 'Yes' : 'No',
                $reg->optin_telephone ? 'Yes' : 'No',
                $reg->optin_letter ? 'Yes' : 'No',
                $reg->tx_status,
                $reg->tx_status_detail,
                $reg->tx_code,
                $reg->tx_sagepay_id,
                $reg->tx_cardtype,
                $reg->tx_card_lastdigits,
            ];

            fputcsv($csv, $row);
        }

        rewind($csv);
        $output = stream_get_contents($csv);

        $response = $this->response;
        $response = $response->withStringBody($output);
        $response = $response->withType('csv');
        $response = $response->withDownload('lights-' . date('Y-m-d-h-i-s') . '.csv');
        return $response;
    }
}
